import os, io, csv, json, zipfile, random, datetime
from django.core.management.base import BaseCommand
import pycountry
from faker import Faker
try:
    import bcrypt
except Exception:
    bcrypt = None

LOCALE_MAP = {
    'IN': 'en_IN', 'US': 'en_US', 'GB': 'en_GB', 'CA': 'en_CA', 'AU': 'en_AU',
    'DE': 'de_DE', 'FR': 'fr_FR', 'ES': 'es_ES', 'IT': 'it_IT', 'JP': 'ja_JP',
    'CN': 'zh_CN', 'BR': 'pt_BR', 'RU': 'ru_RU', 'KR': 'ko_KR', 'MX': 'es_MX',
    'NL': 'nl_NL', 'SE': 'sv_SE', 'NO': 'nb_NO', 'FI': 'fi_FI', 'DK': 'da_DK',
    'PL': 'pl_PL', 'PT': 'pt_PT', 'TR': 'tr_TR', 'AR': 'es_AR', 'BE': 'fr_BE',
    'CH': 'de_CH', 'AT': 'de_AT', 'IE': 'en_IE', 'NZ': 'en_NZ', 'ZA': 'en_ZA',
}
CALLING_CODE = {
    'IN': '91','US': '1','GB': '44','CA': '1','AU': '61','DE': '49','FR': '33','ES': '34','IT': '39','JP': '81',
    'CN': '86','BR': '55','RU': '7','KR': '82','MX': '52','NL': '31','SE': '46','NO': '47','FI': '358','DK': '45',
    'PL': '48','PT': '351','TR': '90','AR': '54','BE': '32','CH': '41','AT': '43','IE': '353','NZ': '64','ZA': '27',
}

def fake_for_country(alpha2):
    locale = LOCALE_MAP.get(alpha2, 'en_US')
    try:
        f = Faker(locale)
    except Exception:
        f = Faker('en_US')
    return f

class Command(BaseCommand):
    help = 'Generate dummy user data and package into ZIP (all countries supported via pycountry)'

    def add_arguments(self, parser):
        parser.add_argument('--count', type=int, default=100)
        parser.add_argument('--country-code', type=str, default='IN', help='ISO alpha-2 country code (e.g., IN, US, GB)')
        parser.add_argument('--output-dir', type=str, default='output')
        parser.add_argument('--hash', action='store_true', help='Hash passwords with bcrypt (requires bcrypt installed)')

    def handle(self, *args, **options):
        count = options['count']
        code = options['country_code'].upper()
        outdir = options['output_dir']
        do_hash = options['hash']
        os.makedirs(outdir, exist_ok=True)
        f = fake_for_country(code)
        records = []
        for i in range(count):
            first = f.first_name()
            last = f.last_name()
            username = (first[0] + last + str(random.randint(1,999))).lower()
            email = f'{first.lower()}.{last.lower()}{random.randint(1,999)}@example.com'
            pwd = f.password(length=12)
            if do_hash and bcrypt:
                password_field = bcrypt.hashpw(pwd.encode('utf-8'), bcrypt.gensalt()).decode('utf-8')
            else:
                password_field = pwd
            address = f.address().replace('\n', ', ')
            city = f.city()
            postal = getattr(f, 'postcode', lambda: '')()
            calling = CALLING_CODE.get(code)
            phone_local = getattr(f, 'phone_number', lambda: '0000')()
            phone = phone_local
            if calling:
                digits = ''.join([c for c in phone_local if c.isdigit()])
                if len(digits) >= 6:
                    phone = f'+{calling}-{digits[-10:]}'
                else:
                    phone = f'+{calling}-{digits}'
            dob = f.date_of_birth(minimum_age=18, maximum_age=75).isoformat() if hasattr(f, 'date_of_birth') else ''
            company = f.company()
            job_title = getattr(f, 'job', lambda: 'Employee')()
            bio = f'{first} {last} works as a {job_title} at {company}.'
            rec = {
                'id': i+1,
                'first_name': first,
                'last_name': last,
                'username': username,
                'email': email,
                'password': password_field,
                'address': address,
                'city': city,
                'state': '',
                'country': code,
                'postal_code': postal,
                'phone': phone,
                'date_of_birth': dob,
                'company': company,
                'job_title': job_title,
                'bio': bio,
                'is_staff': random.choice([True, False, False, False]),
                'is_superuser': False,
                'is_active': True,
                'last_login': datetime.date.today().isoformat(),
                'signup_date': datetime.date.today().isoformat(),
            }
            records.append(rec)
        # write files
        txt_path = os.path.join(outdir, f'users_{code}.txt')
        csv_path = os.path.join(outdir, f'users_{code}.csv')
        with open(txt_path, 'w', encoding='utf-8') as ftxt:
            for r in records:
                ftxt.write(json.dumps(r, ensure_ascii=False) + '\n')
        import csv as _csv
        with open(csv_path, 'w', encoding='utf-8', newline='') as fcsv:
            writer = _csv.DictWriter(fcsv, fieldnames=list(records[0].keys()))
            writer.writeheader()
            writer.writerows(records)
        zip_path = os.path.join(outdir, f'dummy_users_{code}_{count}.zip')
        with zipfile.ZipFile(zip_path, 'w', compression=zipfile.ZIP_DEFLATED) as zf:
            zf.write(txt_path, arcname='users.txt')
            zf.write(csv_path, arcname='users.csv')
            zf.writestr('manifest.json', json.dumps({'num_records': count, 'fields': list(records[0].keys()), 'country': code}, indent=2))
        self.stdout.write(self.style.SUCCESS(f'Generated {count} records into {zip_path}'))
